#include "CRfePurDialog.h"
#include <ui_CRfePurDialog.h>
#include "CRfePurDialog/CRfePurRfSettings.h"

#include <QInputDialog>
#include <QMessageBox>

#include <QrfeSleeper>

#include <reader/QrfeRfePurReader>
#include <QrfeProgressDialog>


CRfePurDialog::CRfePurDialog(QWidget *parent)
    : QDialog(parent)
	, m_reader(0)
{
	m_ui = new Ui::CRfePurDialogClass();
	m_ui->setupUi(this);
	setInfoEdit(m_ui->infoEdit);

	m_rfSettings = new CRfePurRfSettings(m_ui, parent);

	m_ui->antennaSequenceTableWidget->setColumnWidth(0, 25);
	m_ui->antennaSequenceTableWidget->setColumnWidth(1, 125);
	m_ui->antennaSequenceTableWidget->setColumnWidth(2, 120);

	connect(m_ui->okButton, 					SIGNAL(clicked()), 			this, SLOT(accept()));
    connect(m_ui->cancelButton, 				SIGNAL(clicked()), 			this, SLOT(reject()));
	connect(m_ui->refreshButton, 				SIGNAL(clicked()), 			this, SLOT(refresh()));

	connect(m_ui->gpio0InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio1InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio2InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio3InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio4InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio5InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio6InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio7InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio8InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio9InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio10InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio11InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio12InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio13InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio14InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio15InputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );

	connect(m_ui->gpio0OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio1OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio2OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio3OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio4OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio5OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio6OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio7OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio8OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio9OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio10OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio11OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio12OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio13OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio14OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );
	connect(m_ui->gpio15OutputButton, 			SIGNAL(toggled(bool)), 		this, SLOT(setGPIODirection()) );

	connect(m_ui->gpio0LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio1LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio2LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio3LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio4LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio5LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio6LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio7LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio8LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio9LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio10LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio11LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio12LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio13LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio14LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );
	connect(m_ui->gpio15LevelBox, 				SIGNAL(toggled(bool)), 		this, SLOT(setGPIOLevel(bool)) );

    connect(m_ui->antennaSequenceFillButton,	SIGNAL(clicked()), 			this, SLOT(fillAntennaSequence()));
    connect(m_ui->antennaSequenceUpButton,		SIGNAL(clicked()), 			this, SLOT(moveAntennaUp()));
    connect(m_ui->antennaSequenceDownButton,	SIGNAL(clicked()), 			this, SLOT(moveAntennaDown()));
	connect(m_ui->antennaSequenceTableWidget, 	SIGNAL(cellClicked(int,int)),
			this, 								  SLOT(antennaClicked(int,int)) );

	connect(m_ui->rebootButton, 				SIGNAL(clicked()), 			this, SLOT(rebootDevice()));
	connect(m_ui->updateButton, 				SIGNAL(clicked()), 			this, SLOT(updateDevice()));
	connect(m_ui->storeButton, 					SIGNAL(clicked()), 			this, SLOT(storeSettings()));
	connect(m_ui->restoreButton, 				SIGNAL(clicked()), 			this, SLOT(restoreFactory()));
}

CRfePurDialog::~CRfePurDialog()
{
	delete m_ui;
	delete m_rfSettings;
}


int CRfePurDialog::exec(QrfeReaderInterface* ph, bool fullFunction)
{
	m_reader = qobject_cast<QrfeRfePurReader*>(ph);
	if( m_reader == 0 )
		return 0;

	clearInfo();

	m_ui->inventoryModeBox->clear();
	m_ui->inventoryModeBox->addItem("Standard-Multi-Tag", (int)2);
	m_ui->inventoryModeBox->addItem("Fast-Multi-Tag", (int)0);
	m_ui->inventoryModeBox->addItem("Fast-Single-Tag", (int)1);
	m_ui->inventoryModeBox->addItem("NXP Read-Protected Inventory", (int)3);

	m_ui->tagIdHandlerComboBox->clear();
	m_ui->tagIdHandlerComboBox->addItem("Send-Tag-Id Immediately", (int)0);
	m_ui->tagIdHandlerComboBox->addItem("Send-Tag-Id Once", (int)1);

	m_ui->moduleLabel->setText(m_reader->readerModuleString());

	m_ui->readerNameLineEdit->setText(m_reader->readerName());
	m_ui->readerLocationLineEdit->setText(m_reader->readerLocation());

	///////////////////////////////////////////////////////////////
	// Check reader type and activate components
	m_ui->storeButton->setVisible(false);
	m_ui->restoreButton->setVisible(false);

	ulong caps = m_reader->readerCaps();
	if ( (caps & QrfeGlobal::EEPROM_AVAILABLE) != 0 )
	{
		m_ui->storeButton->setVisible(true);
		m_ui->restoreButton->setVisible(true);
	}


	///////////////////////////////////////////////////////////////
	// Check Software revision and activate components
	m_ui->gpioTab->setEnabled(false);
	m_ui->sendFrequencyLabel->setEnabled(false);
	m_ui->sendFrequencyBox->setEnabled(false);
	m_ui->postDetectReadOnGroupBox->setEnabled(false);
	m_ui->powerSafeCountLabel->setEnabled(false);
	m_ui->powerSafeCountSpinBox->setEnabled(false);
	m_ui->powerSafeCountSpinBox->setValue(1);


	ulong sw_rev = m_reader->softwareRevision();
	sw_rev &= 0x00007FFF;
	if(sw_rev >= 0x0031)
	{
		m_ui->gpioTab->setEnabled(true);
	}

	if(sw_rev >= 0x0109)
	{
		m_ui->sendFrequencyLabel->setEnabled(true);
		m_ui->sendFrequencyBox->setEnabled(true);
	}

	if(sw_rev >= 0x0119)
	{
		m_ui->inventoryModeBox->insertItem(1, "Standard-Single-Tag", (int)4);
		m_ui->tagIdHandlerComboBox->addItem("Send-Tag-Id & Stop", (int)2);
	}

	if(sw_rev >= 0x0207)
	{
		m_ui->postDetectReadOnGroupBox->setEnabled(true);
	}

	if(sw_rev >= 0x0212)
	{
		m_ui->powerSafeCountLabel->setEnabled(true);
		m_ui->powerSafeCountSpinBox->setEnabled(true);
	}

	///////////////////////////////////////////////////////////////
	// Check Software antenna count and activate components
	m_ui->antennaTab->setEnabled(false);

	if(m_reader->antennaCount() > 1)
	{
		m_ui->antennaTab->setEnabled(true);
	}

	refresh();

	m_ui->rebootButton->setEnabled(fullFunction);
	m_ui->updateButton->setEnabled(fullFunction);

	connect(m_reader, 			SIGNAL(lostConnection()),
			this, 				  SLOT(reject()) );

	return QDialog::exec();
}

void CRfePurDialog::done(int r)
{
    if(r == QDialog::Accepted)
    {
    	if(!writeAll())return;
    }

	m_reader->setAntennaPower(false);
	m_reader->readSettings();

	disconnect( m_reader, 			SIGNAL(lostConnection()),
				this, 				  SLOT(reject()) );

	QDialog::done(r);
}

bool CRfePurDialog::writeAll()
{
	clearInfo();

	QrfeProgressDialog pb("Writing Settings To Reader...", QrfeGlobal::FLOATING, parentWidget());
	pb.setVisible(true);

	bool ok = writeSettingsToReader(&pb);

	if(!ok)
		showNOK();
	else
		showOK();

	pb.hide();

	//QrfeSleeper::MSleepAlive(100);

	return ok;
}

void CRfePurDialog::refresh()
{
	///////////////////////////////////////////////////////////////
	// Read data from reader
	QrfeProgressDialog pb("Reading Settings From Reader...", QrfeGlobal::FLOATING, parentWidget());
	pb.setVisible(true);

	pb.increasePorgressBar();

	m_reader->setAntennaPower(false);

	pb.increasePorgressBar();

	bool ok = readSettingsFromReader(&pb);

	pb.hide();

	if(ok)
		showOK();
	else
		showNOK();
}


///////////////////////////////////////////////////////////////////////////////////////
// Application
bool CRfePurDialog::setReaderName()
{
	QString name = m_ui->readerNameLineEdit->text();
	if(m_reader->setReaderName(name) == QrfeGlobal::RES_OK){
		infoAppend("OK:  Wrote Reader Name set to " + name);
	}
	else{
		infoAppend("-- Could not set Reader Name to" + name + " --");
		return false;
	}
	return true;
}

bool CRfePurDialog::setReaderLocation()
{
	QString name = m_ui->readerLocationLineEdit->text();
	if(m_reader->setReaderLocation(name) == QrfeGlobal::RES_OK){
		infoAppend("OK:  Wrote Reader Location to " + name );
	}
	else{
		infoAppend("-- Could not set Reader Location to" + name + " --");
		return false;
	}
	return true;
}

bool CRfePurDialog::setInventoryMode()
{
	QString mode = m_ui->inventoryModeBox->currentText();
	int id = m_ui->inventoryModeBox->itemData(m_ui->inventoryModeBox->currentIndex()).toInt();

	if(m_reader->setInventoryMode(id) == QrfeGlobal::RES_OK){
		infoAppend("OK:  Wrote Inventory Mode to " + mode + " (" + QString::number(id) + ")");
	}
	else{
		infoAppend("-- Could not set Inventory Mode to " + mode + " --");
		return false;
	}
	return true;
}

bool CRfePurDialog::setPowerSafe()
{
	QString mode = (m_ui->powerSafeBox->isChecked())?"On":"Off";

	if(m_reader->setPowerSafe(m_ui->powerSafeBox->isChecked(), m_ui->powerSafeIntervalBox->value()) != QrfeGlobal::RES_OK){
		infoAppend("-- Could not set Power Safe " + mode + " with an interval of " + QString::number(m_ui->powerSafeIntervalBox->value()) + " --");
		return false;
	}

	bool on;
	ushort interval = 0;
	if(m_reader->powerSafe(on, interval) != QrfeGlobal::RES_OK){
		infoAppend("-- Could not get Power Safe " + mode + " with an interval of " + QString::number(interval) + " --");
		return false;
	}

	if(on != m_ui->powerSafeBox->isChecked() || interval != m_ui->powerSafeIntervalBox->value()){
		m_ui->powerSafeBox->setChecked(on);
		m_ui->powerSafeIntervalBox->setValue(interval);
		infoAppend("-- Could not set Power Safe " + mode + " with an interval of " + QString::number(interval) + " --");
		return false;
	}


	if(m_ui->powerSafeCountLabel->isEnabled())
	{
		ushort count = 0;

		if(m_reader->setPowerSafeCount(m_ui->powerSafeCountSpinBox->value()) != QrfeGlobal::RES_OK){
			infoAppend("-- Could not set Power Safe Count of " + QString::number(m_ui->powerSafeCountSpinBox->value()) + " --");
			return false;
		}

		if(m_reader->powerSafeCount(count) != QrfeGlobal::RES_OK){
			infoAppend("-- Could not get Power Safe Count --");
			return false;
		}

		if(count != m_ui->powerSafeCountSpinBox->value())
		{
			m_ui->powerSafeCountSpinBox->setValue(count);
			infoAppend("-- Could not set Power Safe " + mode + " with an interval of " + QString::number(interval) + " --");
			return false;
		}

	}


	infoAppend("OK:  Wrote Power Safe " + mode + " with an interval of " + QString::number(interval));
	return true;
}

bool CRfePurDialog::setSendRSSI()
{
	QString mode = (m_ui->sendRSSIBox->isChecked())?"On":"Off";

	if(m_reader->setRssiEnabled(m_ui->sendRSSIBox->isChecked()) == QrfeGlobal::RES_OK){
		infoAppend("OK:  Wrote Send RSSI set " + mode);
	}
	else{
		infoAppend("-- Could not turn RSSI " + mode + " --");
		return false;
	}
	return true;
}

bool CRfePurDialog::setSendFrequency()
{
	QString mode = (m_ui->sendFrequencyBox->isChecked())?"On":"Off";

	if(m_reader->setReadFrequencyEnabled(m_ui->sendFrequencyBox->isChecked()) == QrfeGlobal::RES_OK){
		infoAppend("OK:  Wrote Send Frequency " + mode);
	}
	else{
		infoAppend("-- Could not turn Frequency " + mode + " --");
		return false;
	}
	return true;
}

bool CRfePurDialog::setTagIdHandler()
{
	QString mode = m_ui->tagIdHandlerComboBox->currentText();
	int id = m_ui->tagIdHandlerComboBox->itemData(m_ui->tagIdHandlerComboBox->currentIndex()).toInt();

	if(m_reader->setTagIdHandler(id) == QrfeGlobal::RES_OK){
		infoAppend("OK:  Wrote Tag id handler to " + mode + " (" + QString::number(id) + ")");
	}
	else{
		infoAppend("-- Could not set tag id handler " + mode + " --");
		return false;
	}
	return true;
}

bool CRfePurDialog::setPostDetectReadParams()
{
	bool on = m_ui->postDetectReadOnGroupBox->isChecked();
	uchar memBank = m_ui->postDetectReadBankComboBox->currentIndex();
	ushort address = m_ui->postDetectReadAddressSpinBox->value();
	uchar size = m_ui->postDetectReadSizeSpinBox->value() * 2;
	bool ok = false;
	QByteArray passw = QrfeGlobal::stringToBytes(m_ui->postDetectReadPasswordEdit->text(), &ok);
	if(!ok)
	{
		QMessageBox::critical(this, "Error", "The password for Post Detect Read is not valid!");
		return false;
	}

	if(m_reader->setPostDetectReadParams(on, memBank, address, size, passw) == QrfeGlobal::RES_OK){
		infoAppend("OK:  Wrote Post Detect Read Params");
	}
	else{
		infoAppend("-- Could not set Post Detect Read Params --");
		return false;
	}
	return true;
}




///////////////////////////////////////////////////////////////////////////////////////
// GPIO
bool CRfePurDialog::readGPIOInfos()
{
	m_reloadGPIODirection = false;
    m_reloadGPIOValues = false;

	ulong mask = 0, output = 0, input = 0;
	if(m_reader->getGPIOCaps(mask, output, input) == QrfeGlobal::RES_OK)
	{
		m_ui->gpio0Box->setEnabled((mask & (1 << 0)) != 0);
		m_ui->gpio1Box->setEnabled((mask & (1 << 1)) != 0);
		m_ui->gpio2Box->setEnabled((mask & (1 << 2)) != 0);
		m_ui->gpio3Box->setEnabled((mask & (1 << 3)) != 0);
		m_ui->gpio4Box->setEnabled((mask & (1 << 4)) != 0);
		m_ui->gpio5Box->setEnabled((mask & (1 << 5)) != 0);
		m_ui->gpio6Box->setEnabled((mask & (1 << 6)) != 0);
		m_ui->gpio7Box->setEnabled((mask & (1 << 7)) != 0);
		m_ui->gpio8Box->setEnabled((mask & (1 << 8)) != 0);
		m_ui->gpio9Box->setEnabled((mask & (1 << 9)) != 0);
		m_ui->gpio10Box->setEnabled((mask & (1 << 10)) != 0);
		m_ui->gpio11Box->setEnabled((mask & (1 << 11)) != 0);
		m_ui->gpio12Box->setEnabled((mask & (1 << 12)) != 0);
		m_ui->gpio13Box->setEnabled((mask & (1 << 13)) != 0);
		m_ui->gpio14Box->setEnabled((mask & (1 << 14)) != 0);
		m_ui->gpio15Box->setEnabled((mask & (1 << 15)) != 0);

		m_ui->gpio0InputButton->setEnabled((input & (1 << 0)) != 0);
		m_ui->gpio1InputButton->setEnabled((input & (1 << 1)) != 0);
		m_ui->gpio2InputButton->setEnabled((input & (1 << 2)) != 0);
		m_ui->gpio3InputButton->setEnabled((input & (1 << 3)) != 0);
		m_ui->gpio4InputButton->setEnabled((input & (1 << 4)) != 0);
		m_ui->gpio5InputButton->setEnabled((input & (1 << 5)) != 0);
		m_ui->gpio6InputButton->setEnabled((input & (1 << 6)) != 0);
		m_ui->gpio7InputButton->setEnabled((input & (1 << 7)) != 0);
		m_ui->gpio8InputButton->setEnabled((input & (1 << 8)) != 0);
		m_ui->gpio9InputButton->setEnabled((input & (1 << 9)) != 0);
		m_ui->gpio10InputButton->setEnabled((input & (1 << 10)) != 0);
		m_ui->gpio11InputButton->setEnabled((input & (1 << 11)) != 0);
		m_ui->gpio12InputButton->setEnabled((input & (1 << 12)) != 0);
		m_ui->gpio13InputButton->setEnabled((input & (1 << 13)) != 0);
		m_ui->gpio14InputButton->setEnabled((input & (1 << 14)) != 0);
		m_ui->gpio15InputButton->setEnabled((input & (1 << 15)) != 0);

		m_ui->gpio0OutputButton->setEnabled((output & (1 << 0)) != 0);
		m_ui->gpio1OutputButton->setEnabled((output & (1 << 1)) != 0);
		m_ui->gpio2OutputButton->setEnabled((output & (1 << 2)) != 0);
		m_ui->gpio3OutputButton->setEnabled((output & (1 << 3)) != 0);
		m_ui->gpio4OutputButton->setEnabled((output & (1 << 4)) != 0);
		m_ui->gpio5OutputButton->setEnabled((output & (1 << 5)) != 0);
		m_ui->gpio6OutputButton->setEnabled((output & (1 << 6)) != 0);
		m_ui->gpio7OutputButton->setEnabled((output & (1 << 7)) != 0);
		m_ui->gpio8OutputButton->setEnabled((output & (1 << 8)) != 0);
		m_ui->gpio9OutputButton->setEnabled((output & (1 << 9)) != 0);
		m_ui->gpio10OutputButton->setEnabled((output & (1 << 10)) != 0);
		m_ui->gpio11OutputButton->setEnabled((output & (1 << 11)) != 0);
		m_ui->gpio12OutputButton->setEnabled((output & (1 << 12)) != 0);
		m_ui->gpio13OutputButton->setEnabled((output & (1 << 13)) != 0);
		m_ui->gpio14OutputButton->setEnabled((output & (1 << 14)) != 0);
		m_ui->gpio15OutputButton->setEnabled((output & (1 << 15)) != 0);
	}
	else{
		infoSetError("ERR: Read GPIO data");
		m_ui->gpioTab->setEnabled(false);
		return false;
	}

	reloadGPIODirection();
	reloadGPIOValues();



	infoAppend("OK:  Read GPIO data");

	return true;
}

void CRfePurDialog::setGPIOLevel(bool high)
{
    if(m_reloadGPIOValues)
        return;

	ulong mask = 0;
	if( ((QObject*)m_ui->gpio0LevelBox) == sender())
		mask = 1 << 0;
	else if( ((QObject*)m_ui->gpio1LevelBox) == sender())
		mask = 1 << 1;
	else if( ((QObject*)m_ui->gpio2LevelBox) == sender())
		mask = 1 << 2;
	else if( ((QObject*)m_ui->gpio3LevelBox) == sender())
		mask = 1 << 3;
	else if( ((QObject*)m_ui->gpio4LevelBox) == sender())
		mask = 1 << 4;
	else if( ((QObject*)m_ui->gpio5LevelBox) == sender())
		mask = 1 << 5;
	else if( ((QObject*)m_ui->gpio6LevelBox) == sender())
		mask = 1 << 6;
	else if( ((QObject*)m_ui->gpio7LevelBox) == sender())
		mask = 1 << 7;
	else if( ((QObject*)m_ui->gpio8LevelBox) == sender())
		mask = 1 << 8;
	else if( ((QObject*)m_ui->gpio9LevelBox) == sender())
		mask = 1 << 9;
	else if( ((QObject*)m_ui->gpio10LevelBox) == sender())
		mask = 1 << 10;
	else if( ((QObject*)m_ui->gpio11LevelBox) == sender())
		mask = 1 << 11;
	else if( ((QObject*)m_ui->gpio12LevelBox) == sender())
		mask = 1 << 12;
	else if( ((QObject*)m_ui->gpio13LevelBox) == sender())
		mask = 1 << 13;
	else if( ((QObject*)m_ui->gpio14LevelBox) == sender())
		mask = 1 << 14;
	else if( ((QObject*)m_ui->gpio15LevelBox) == sender())
		mask = 1 << 15;

	clearInfo();
	QrfeGlobal::Result res;
	if(high)
		res = m_reader->setGPIO(mask);
	else
		res = m_reader->clearGPIO(mask);

	if(res == QrfeGlobal::RES_OK){
		if(high)
			infoSetOK("-- Set GPIO mask " + QString("0x%1").arg(mask, 2, 16, QChar('0')) + " --");
		else
			infoSetOK("-- Cleared GPIO mask " + QString("0x%1").arg(mask, 2, 16, QChar('0')) + " --");

		showOK();
	}
	else
	{
		if(high)
			infoSetError("-- Could not set GPIO mask " + QString("0x%1").arg(mask, 2, 16, QChar('0')) + " --");
		else
			infoSetError("-- Could not clear GPIO mask " + QString("0x%1").arg(mask, 2, 16, QChar('0')) + " --");

		qobject_cast<QCheckBox*>(sender())->setChecked(!high);
		showNOK();
	}

    reloadGPIODirection();
    reloadGPIOValues();
}

void CRfePurDialog::setGPIODirection()
{
	if(m_reloadGPIODirection)
		return;

	ulong mask = 0;
	if( m_ui->gpio0OutputButton->isChecked())
		mask |= 1 << 0;
	if( m_ui->gpio1OutputButton->isChecked())
		mask |= 1 << 1;
	if( m_ui->gpio2OutputButton->isChecked())
		mask |= 1 << 2;
	if( m_ui->gpio3OutputButton->isChecked())
		mask |= 1 << 3;
	if( m_ui->gpio4OutputButton->isChecked())
		mask |= 1 << 4;
	if( m_ui->gpio5OutputButton->isChecked())
		mask |= 1 << 5;
	if( m_ui->gpio6OutputButton->isChecked())
		mask |= 1 << 6;
	if( m_ui->gpio7OutputButton->isChecked())
		mask |= 1 << 7;
	if( m_ui->gpio8OutputButton->isChecked())
		mask |= 1 << 8;
	if( m_ui->gpio9OutputButton->isChecked())
		mask |= 1 << 9;
	if( m_ui->gpio10OutputButton->isChecked())
		mask |= 1 << 10;
	if( m_ui->gpio11OutputButton->isChecked())
		mask |= 1 << 11;
	if( m_ui->gpio12OutputButton->isChecked())
		mask |= 1 << 12;
	if( m_ui->gpio13OutputButton->isChecked())
		mask |= 1 << 13;
	if( m_ui->gpio14OutputButton->isChecked())
		mask |= 1 << 14;
	if( m_ui->gpio15OutputButton->isChecked())
		mask |= 1 << 15;

	clearInfo();
	QrfeGlobal::Result res = QrfeGlobal::RES_OK;
	res = m_reader->setGPIODirection(mask);

	if(res == QrfeGlobal::RES_OK){
		infoSetOK("-- Set GPIO direction " + QString("0x%1").arg(mask, 2, 16, QChar('0')) + " --");
		showOK();
	}
	else
	{
		infoSetError("-- Could not set GPIO direction " + QString("0x%1").arg(mask, 2, 16, QChar('0')) + " --");
		showNOK();
	}

	reloadGPIODirection();
	reloadGPIOValues();
}

void CRfePurDialog::reloadGPIODirection()
{
	m_reloadGPIODirection = true;

	ulong mask;
	if(m_reader->getGPIODirection(mask) == QrfeGlobal::RES_OK)
	{
		ulong input = ~mask;
		m_ui->gpio0InputButton->setChecked((input & (1 << 0)) != 0);
		m_ui->gpio1InputButton->setChecked((input & (1 << 1)) != 0);
		m_ui->gpio2InputButton->setChecked((input & (1 << 2)) != 0);
		m_ui->gpio3InputButton->setChecked((input & (1 << 3)) != 0);
		m_ui->gpio4InputButton->setChecked((input & (1 << 4)) != 0);
		m_ui->gpio5InputButton->setChecked((input & (1 << 5)) != 0);
		m_ui->gpio6InputButton->setChecked((input & (1 << 6)) != 0);
		m_ui->gpio7InputButton->setChecked((input & (1 << 7)) != 0);
		m_ui->gpio8InputButton->setChecked((input & (1 << 8)) != 0);
		m_ui->gpio9InputButton->setChecked((input & (1 << 9)) != 0);
		m_ui->gpio10InputButton->setChecked((input & (1 << 10)) != 0);
		m_ui->gpio11InputButton->setChecked((input & (1 << 11)) != 0);
		m_ui->gpio12InputButton->setChecked((input & (1 << 12)) != 0);
		m_ui->gpio13InputButton->setChecked((input & (1 << 13)) != 0);
		m_ui->gpio14InputButton->setChecked((input & (1 << 14)) != 0);
		m_ui->gpio15InputButton->setChecked((input & (1 << 15)) != 0);

		ulong output = mask;
		m_ui->gpio0OutputButton->setChecked((output & (1 << 0)) != 0);
		m_ui->gpio1OutputButton->setChecked((output & (1 << 1)) != 0);
		m_ui->gpio2OutputButton->setChecked((output & (1 << 2)) != 0);
		m_ui->gpio3OutputButton->setChecked((output & (1 << 3)) != 0);
		m_ui->gpio4OutputButton->setChecked((output & (1 << 4)) != 0);
		m_ui->gpio5OutputButton->setChecked((output & (1 << 5)) != 0);
		m_ui->gpio6OutputButton->setChecked((output & (1 << 6)) != 0);
		m_ui->gpio7OutputButton->setChecked((output & (1 << 7)) != 0);
		m_ui->gpio8OutputButton->setChecked((output & (1 << 8)) != 0);
		m_ui->gpio9OutputButton->setChecked((output & (1 << 9)) != 0);
		m_ui->gpio10OutputButton->setChecked((output & (1 << 10)) != 0);
		m_ui->gpio11OutputButton->setChecked((output & (1 << 11)) != 0);
		m_ui->gpio12OutputButton->setChecked((output & (1 << 12)) != 0);
		m_ui->gpio13OutputButton->setChecked((output & (1 << 13)) != 0);
		m_ui->gpio14OutputButton->setChecked((output & (1 << 14)) != 0);
		m_ui->gpio15OutputButton->setChecked((output & (1 << 15)) != 0);

		m_ui->gpio0LevelBox->setEnabled((output & (1 << 0)) != 0);
		m_ui->gpio1LevelBox->setEnabled((output & (1 << 1)) != 0);
		m_ui->gpio2LevelBox->setEnabled((output & (1 << 2)) != 0);
		m_ui->gpio3LevelBox->setEnabled((output & (1 << 3)) != 0);
		m_ui->gpio4LevelBox->setEnabled((output & (1 << 4)) != 0);
		m_ui->gpio5LevelBox->setEnabled((output & (1 << 5)) != 0);
		m_ui->gpio6LevelBox->setEnabled((output & (1 << 6)) != 0);
		m_ui->gpio7LevelBox->setEnabled((output & (1 << 7)) != 0);
		m_ui->gpio8LevelBox->setEnabled((output & (1 << 8)) != 0);
		m_ui->gpio9LevelBox->setEnabled((output & (1 << 9)) != 0);
		m_ui->gpio10LevelBox->setEnabled((output & (1 << 10)) != 0);
		m_ui->gpio11LevelBox->setEnabled((output & (1 << 11)) != 0);
		m_ui->gpio12LevelBox->setEnabled((output & (1 << 12)) != 0);
		m_ui->gpio13LevelBox->setEnabled((output & (1 << 13)) != 0);
		m_ui->gpio14LevelBox->setEnabled((output & (1 << 14)) != 0);
		m_ui->gpio15LevelBox->setEnabled((output & (1 << 15)) != 0);
	}
	else{
		infoSetError("-- Could not get GPIO direction --");
		m_ui->gpioTab->setEnabled(false);
	}

	m_reloadGPIODirection = false;
}

void CRfePurDialog::reloadGPIOValues()
{
    m_reloadGPIOValues = true;

    ulong mask = 0;
    if(m_reader->getGPIO(mask) == QrfeGlobal::RES_OK)
	{
		m_ui->gpio0LevelBox->setChecked((mask & (1 << 0)) != 0);
		m_ui->gpio1LevelBox->setChecked((mask & (1 << 1)) != 0);
		m_ui->gpio2LevelBox->setChecked((mask & (1 << 2)) != 0);
		m_ui->gpio3LevelBox->setChecked((mask & (1 << 3)) != 0);
		m_ui->gpio4LevelBox->setChecked((mask & (1 << 4)) != 0);
		m_ui->gpio5LevelBox->setChecked((mask & (1 << 5)) != 0);
		m_ui->gpio6LevelBox->setChecked((mask & (1 << 6)) != 0);
		m_ui->gpio7LevelBox->setChecked((mask & (1 << 7)) != 0);
		m_ui->gpio8LevelBox->setChecked((mask & (1 << 8)) != 0);
		m_ui->gpio9LevelBox->setChecked((mask & (1 << 9)) != 0);
		m_ui->gpio10LevelBox->setChecked((mask & (1 << 10)) != 0);
		m_ui->gpio11LevelBox->setChecked((mask & (1 << 11)) != 0);
		m_ui->gpio12LevelBox->setChecked((mask & (1 << 12)) != 0);
		m_ui->gpio13LevelBox->setChecked((mask & (1 << 13)) != 0);
		m_ui->gpio14LevelBox->setChecked((mask & (1 << 14)) != 0);
		m_ui->gpio15LevelBox->setChecked((mask & (1 << 15)) != 0);
	}
	else{
		infoSetError("-- Could not get GPIO direction --");
		m_ui->gpioTab->setEnabled(false);
	}

    m_reloadGPIOValues = false;
}




///////////////////////////////////////////////////////////////////////////////////////
// Antenna
bool CRfePurDialog::readAntennaSequence()
{
	uchar antennaCount = m_reader->antennaCount();

	m_antennaSequence.clear();

	QList<uchar> antennas;
	for(int i = 1; i <= antennaCount; i++)
		antennas << i;

	QList< QPair<uchar,ulong> > sequence;

	if(m_reader->getAntennaSequence(sequence) != QrfeGlobal::RES_OK)
	{
		infoSetError("-- Could not read Antenna Sequence  --");
		return false;
	}

	for(int i = 0; i < sequence.size(); i++)
	{
		AntennaSequenceInfos infos;
		infos.antennaNr = sequence.at(i).first;
		infos.on = true;
		infos.time = sequence.at(i).second;
		m_antennaSequence.append(infos);
		antennas.removeAll(sequence.at(i).first);
	}
	for(int i = 0; i < antennas.size(); i++)
	{
		AntennaSequenceInfos infos;
		infos.antennaNr = antennas.at(i);
		infos.on = false;
		infos.time = 0;
		m_antennaSequence.append(infos);
	}

	reloadAntennaSequence();

	return true;
}

void CRfePurDialog::reloadAntennaSequence()
{
	m_ui->antennaSequenceTableWidget->clearContents();
	m_ui->antennaSequenceTableWidget->setRowCount(0);
	foreach(AntennaSequenceGuiElems a, m_antennaSequenceGuiElems)
	{
		delete a.on;
		delete a.time;
	}
	m_antennaSequenceGuiElems.clear();

	for(int i = 0; i < m_antennaSequence.size(); i++)
	{
		insertAntenna(m_antennaSequence.at(i).antennaNr, m_antennaSequence.at(i).on, m_antennaSequence.at(i).time);
	}

	m_ui->antennaSequenceUpButton->setEnabled(false);
	m_ui->antennaSequenceDownButton->setEnabled(false);
}

void CRfePurDialog::insertAntenna(uchar antennaNr, bool on, uint time)
{
	AntennaSequenceGuiElems elems;
 
	int index = m_ui->antennaSequenceTableWidget->rowCount();

	elems.on = new QWidget(m_ui->antennaSequenceTableWidget);
	elems.onBox = new QCheckBox(elems.on);
	elems.onBox->setChecked(on);
	elems.onBox->setProperty("Index", index);
	elems.onBox->setProperty("AntennaNumber", antennaNr);
	connect(elems.onBox, 	SIGNAL(toggled( bool )),
			this, 			  SLOT(antennaWasChanged( bool )) );

	QHBoxLayout *onLayout = new QHBoxLayout(elems.on);
	onLayout->setContentsMargins ( 0, 0, 0, 0);
	onLayout->addStretch(); 
	onLayout->addWidget(elems.onBox);
	onLayout->addStretch();
	elems.on->setFixedHeight(17);


	QTableWidgetItem* antenna = new QTableWidgetItem("Antenna #" + QString::number(antennaNr));
	antenna->setTextAlignment(Qt::AlignCenter);

	elems.time = new QWidget(m_ui->antennaSequenceTableWidget);
	elems.timeBox = new QSpinBox(elems.time);
	elems.timeBox->setMaximum(0x7FFFFFFF);
	elems.timeBox->setSuffix("ms");
	elems.timeBox->setValue(time);
	elems.timeBox->setProperty("Index", index);
	elems.timeBox->setProperty("AntennaNumber", antennaNr);
	connect(elems.timeBox, 	SIGNAL( valueChanged ( int )),
			this, 			  SLOT( antennaTimeChanged( int  )) );

	QHBoxLayout *timeLayout = new QHBoxLayout(elems.time);
	timeLayout->setContentsMargins ( 0, 0, 0, 0);
	timeLayout->addWidget(elems.timeBox);
	elems.time->setFixedHeight(17);

	m_antennaSequenceGuiElems.append(elems);

	m_ui->antennaSequenceTableWidget->setRowCount(index+1);

	m_ui->antennaSequenceTableWidget->setCellWidget(index, 0, elems.on);
	m_ui->antennaSequenceTableWidget->setItem	   (index, 1, antenna );
	m_ui->antennaSequenceTableWidget->setCellWidget(index, 2, elems.time);

	if(!on)
	{
		antenna->setBackgroundColor(Qt::lightGray);

		QPalette p = elems.on->palette();
		p.setColor(QPalette::Base, Qt::lightGray);

		elems.on->setAutoFillBackground(true);
		elems.on->setPalette(p);
		elems.time->setEnabled(false);
	}

	m_ui->antennaSequenceTableWidget->setRowHeight(index, 17);
}

void CRfePurDialog::antennaWasChanged(bool on)
{
	QCheckBox* cb = qobject_cast<QCheckBox*>(sender());
	if(cb == 0)
		return; 

	QVariant p = cb->property("Index");
	if(!p.isValid())
		return;

	int index = p.toInt();
	if(index > m_ui->antennaSequenceTableWidget->rowCount())
		return;

	m_antennaSequence[index].on = on;
	if(on)
	{
		//bool ok = false;
        //int time = QInputDialog::getInt(this, tr("Activate Antenna #") + QString::number(index), "How long should the antenna #" + QString::number(index) + " be active? (ms)", 2000, 1, 0x7FFFFFFF, 500, &ok);
		
		//if(!ok)
		//{
		//	cb->setChecked(false);
		//	return;
		//}
		int time = m_ui->antennaSequenceDefaultTimeSpinBox->value();

		m_antennaSequence[index].time = time;
        AntennaSequenceInfos sequence = m_antennaSequence.takeAt(index);

        bool inserted = false;
        for(int i = 0; i < m_antennaSequence.size(); i++)
        {
            if(m_antennaSequence[i].on == false){
                m_antennaSequence.insert(i, sequence);
                inserted = true;
                break;
            }
        }
        if(!inserted)
            m_antennaSequence.push_back(sequence);

        for(int i = 0; i < m_antennaSequence.size(); i++)
        {
            if(m_antennaSequence[i].on && (m_antennaSequence[i].time == 0 || m_antennaSequence[i].time == 0xFFFFFFFF)){
                m_antennaSequence[i].time = time;
            }
        }
    }
	else
	{
		m_antennaSequence[index].time = 0;

        AntennaSequenceInfos sequence = m_antennaSequence.takeAt(index);

        bool inserted = false;
        for(int i = index; i < m_antennaSequence.size(); i++)
        {
            if(m_antennaSequence[i].on == false){
                m_antennaSequence.insert(i, sequence);
                inserted = true;
                break;
            }
        }
        if(!inserted)
            m_antennaSequence.push_back(sequence);
	}

	QTimer::singleShot(0, this, SLOT(reloadAntennaSequence()) );
}

void CRfePurDialog::antennaTimeChanged ( int time )
{
	QSpinBox* cb = qobject_cast<QSpinBox*>(sender());
	if(cb == 0)
		return;

	QVariant p = cb->property("Index");
	if(!p.isValid())
		return;

	int index = p.toInt();
	if(index > m_ui->antennaSequenceTableWidget->rowCount())
		return;

	m_antennaSequence[index].time = time;
    //QTimer::singleShot(0, this, SLOT(reloadAntennaSequence()) );
}

void CRfePurDialog::antennaClicked ( int row, int /*column*/ )
{
	m_ui->antennaSequenceUpButton->setEnabled(false);
	m_ui->antennaSequenceDownButton->setEnabled(false);

	if(m_antennaSequence.at(row).on)
	{
		if(row > 0)
			m_ui->antennaSequenceUpButton->setEnabled(true);

		if(row < (m_antennaSequence.size()-1) && m_antennaSequence.at(row+1).on == true)
			m_ui->antennaSequenceDownButton->setEnabled(true);
	}
}

void CRfePurDialog::moveAntennaUp()
{
	int index = m_ui->antennaSequenceTableWidget->currentRow();

	if(!m_antennaSequence[index].on)
		return;

	if(index == 0)
		return;

	AntennaSequenceInfos elem = m_antennaSequence.takeAt(index);
	m_antennaSequence.insert(index-1, elem);

	reloadAntennaSequence();

	m_ui->antennaSequenceTableWidget->setCurrentCell(index-1, 1);
	m_ui->antennaSequenceTableWidget->setFocus(Qt::TabFocusReason);
	antennaClicked(index-1, 1);
}

void CRfePurDialog::moveAntennaDown()
{
	int index = m_ui->antennaSequenceTableWidget->currentRow();

	if(!m_antennaSequence[index].on)
		return;

	if(index == (m_antennaSequence.size()-1))
		return;

	if(m_antennaSequence.at(index+1).on != true)
		return;

	AntennaSequenceInfos elem = m_antennaSequence.takeAt(index);
	m_antennaSequence.insert(index+1, elem);

	reloadAntennaSequence();

	m_ui->antennaSequenceTableWidget->setCurrentCell(index+1, 1);
	m_ui->antennaSequenceTableWidget->setFocus(Qt::TabFocusReason);
    antennaClicked(index+1, 1);
}

void CRfePurDialog::fillAntennaSequence()
{
    int time = QInputDialog::getInt(this, "Processing Time", "Enter processing time for each antenna port (ms):", 1000, 1);

    for(int i = 0; i < m_antennaSequence.size(); i++)
    {
        if(m_antennaSequence[i].on)
            m_antennaSequence[i].time = time;
    }

    reloadAntennaSequence();
}

bool CRfePurDialog::setAntennaSequence()
{
	QList< QPair<uchar,ulong> > sequence;
	for(int i = 0; i < m_antennaSequence.size(); i++)
	{
		if( m_antennaSequence.at(i).on )
		{
			QPair<uchar,ulong> p;
			p.first   = m_antennaSequence.at(i).antennaNr;
			p.second  = m_antennaSequence.at(i).time;
			sequence.append(p);
		}
	}

	if(m_reader->setAntennaSequence(sequence) == QrfeGlobal::RES_OK)
	{
		infoAppend("OK:  Wrote antenna sequence");
	}
	else
	{
		infoAppend("-- Could not set Antenna Sequence --");
		return false;
	}

	QTimer::singleShot(0, this, SLOT(readAntennaSequence()) );
	return true;
}



void CRfePurDialog::storeSettings()
{
	clearInfo();
	if(!writeAll())
		return;
	if(m_reader->saveSettingsPermanent() == QrfeGlobal::RES_OK)
	{
		showOK();
		infoAppend("-- Settings stored  --");
	}
	else
	{
		infoSetError("-- Could not store settings --");
	}

}

void CRfePurDialog::restoreFactory()
{
	clearInfo();
	if(m_reader->restoreFactorySettings() == QrfeGlobal::RES_OK){
		infoSetOK("-- Factory settings restored  --");

		refresh();
	}
	else{
		infoSetError("-- Could not restore factory settings --");
	}

}



void CRfePurDialog::rebootDevice()
{
	clearInfo();

	QString 				readerId	= m_reader->readerId();
	QrfeGlobal::DeviceType	deviceType 	= m_reader->deviceType();
	QString 				deviceName 	= m_reader->deviceName();

	if(m_reader->reboot() == QrfeGlobal::RES_OK){
		m_reader->destroy();
		infoSet("-- The device will reboot --");

		QList<QVariant> params;
		params.append(QString("Rebooting device..."));
		params.append((uint)2500);
		params.append(QVariant::fromValue(qobject_cast<QWidget*>(parent())));
		emit appendGlobalJob(JOB_WaitForSpeciifedTimeWithDialog, params);

		params.clear();
		params.append(deviceType);
		params.append(deviceName);
		emit appendGlobalJob(JOB_AddReader, params);

		params.clear();
		params.append(readerId);
		emit appendGlobalJob(JOB_ShowSettingsOfReader, params);

        QDialog::reject();
	}
	else{
		infoSet("-- Could not reboot device --");
		showNOK();
		return;
	}
}

void CRfePurDialog::updateDevice()
{
	clearInfo();

	QList<QVariant> params;
	params.append(m_reader->readerId());
	emit appendGlobalJob(JOB_SwitchReaderToBootloaderAndShowDialog, params);

	accept();
}



bool CRfePurDialog::writeSettingsToReader( QrfeProgressDialog* pb)
{
	bool ok = false;
	ok = setReaderName();
	pb->increasePorgressBar();

	ok = setReaderLocation();
	pb->increasePorgressBar();

	ok = setInventoryMode();
	pb->increasePorgressBar();

	ok = setPowerSafe();
	pb->increasePorgressBar();

	ok = setSendRSSI();
	pb->increasePorgressBar();

	ok = setSendFrequency();
	pb->increasePorgressBar();

	ok = setTagIdHandler();
	pb->increasePorgressBar();

	ok = setPostDetectReadParams();
	pb->increasePorgressBar();

	ok = setAntennaSequence();
	pb->increasePorgressBar();

	ok = m_rfSettings->writeSettingsToReader(pb);

	return ok;
}

bool CRfePurDialog::readSettingsFromReader(QrfeProgressDialog* pb)
{
	bool err = false;

	pb->increasePorgressBar();

	ushort inventoryMode;
	if(m_reader->inventoryMode(inventoryMode) == QrfeGlobal::RES_OK)
	{
		m_ui->inventoryModeBox->setCurrentIndex(m_ui->inventoryModeBox->findData((int)inventoryMode));
		infoAppend("OK:  Read inventory mode");
	}
	else
	{
		infoAppend("ERR: Read inventory mode");
		err = true;
	}

	pb->increasePorgressBar();

	bool powerSafeOn = false;
	ushort interval = 0;
	if(m_reader->powerSafe(powerSafeOn, interval) == QrfeGlobal::RES_OK){
		m_ui->powerSafeBox->setChecked(powerSafeOn);
		m_ui->powerSafeIntervalBox->setValue(interval);
		infoAppend("OK:  Read power safe");
	}
	else
	{
		infoAppend("ERR: Read power safe");
		err = true;
	}

	if(m_ui->powerSafeCountLabel->isEnabled())
	{
		ushort count = 0;
		if(m_reader->powerSafeCount(count) == QrfeGlobal::RES_OK){
			m_ui->powerSafeCountSpinBox->setValue(count);
			infoAppend("OK:  Read power safe count");
		}
		else
		{
			infoAppend("ERR: Read power safe count");
			err = true;
		}
	}

	pb->increasePorgressBar();

	bool rssiEnabled = 0;
	if(m_reader->rssiEnabled(rssiEnabled) == QrfeGlobal::RES_OK){
		m_ui->sendRSSIBox->setChecked(rssiEnabled);
		infoAppend("OK:  Read send rssi");
	}
	else
	{
		infoAppend("ERR: Read send rssi");
		err = true;
	}

	pb->increasePorgressBar();

	uchar tagIdHandler = 0;
	if(m_reader->tagIdHandler(tagIdHandler) == QrfeGlobal::RES_OK)
	{
		m_ui->tagIdHandlerComboBox->setCurrentIndex(m_ui->tagIdHandlerComboBox->findData((int)tagIdHandler));
		infoAppend("OK:  Read tag id handler");
	}
	else
	{
		infoAppend("ERR: Read tag id handler");
		err = true;
	}

	pb->increasePorgressBar();

	if(m_ui->sendFrequencyBox->isEnabled())
	{
		bool sendFrequency = 0;
		if(m_reader->readFrequencyEnabled(sendFrequency) == QrfeGlobal::RES_OK){
			m_ui->sendFrequencyBox->setChecked(sendFrequency);
			infoAppend("OK:  Read send frequency");
		}
	}
	else
	{
		infoAppend("ERR: Read send frequency");
		err = true;
	}


	pb->increasePorgressBar();

	if(m_ui->postDetectReadOnGroupBox->isEnabled())
	{
		bool on;
		uchar memBank;
		ushort address;
		uchar size;
		QByteArray password;
		if(m_reader->postDetectReadParams(on, memBank, address, size, password) == QrfeGlobal::RES_OK){
			m_ui->postDetectReadOnGroupBox->setChecked(on);
			m_ui->postDetectReadBankComboBox->setCurrentIndex(memBank);
			m_ui->postDetectReadAddressSpinBox->setValue(address);
			m_ui->postDetectReadSizeSpinBox->setValue(size/2);
			m_ui->postDetectReadPasswordEdit->setText(QrfeGlobal::bytesToString(password));
			infoAppend("OK:  Read post detect read");
		}
		else
		{
			infoAppend("ERR: Read post detect read");
			err = true;
		}
	}

	pb->increasePorgressBar();

	if(!readGPIOInfos())
		err = true;

	pb->increasePorgressBar();

	if(!readAntennaSequence())
		err = true;

	pb->increasePorgressBar();

	if(!m_rfSettings->init(m_reader, pb))
		err = true;

	return !err;
}



